import java.util.ArrayList;
import java.util.LinkedList;

// Class to parse a specific set of commands
public class CmdParser {
	BSTree<String, CityRecord> GISTree;	// Binary Search Tree
	PRQuadTree CTree;					// PR QuadTree
	
	public CmdParser() {
		// Initialize BSTree and PRQuadTree
		GISTree = new BSTree<String, CityRecord>();
		CTree = new PRQuadTree();
	}
	
	// Calls specific functions depending on input
	public void Parse(String input) {
		// Split the passed line into tokens
		String delim = "[ ]+";
		String[] tokens = input.split(delim);
		
		// Echo the command to the output
		for (String token : tokens) {
			System.out.printf(token + " ");
		}
		System.out.printf("\n");
		
		// Parse tokens to determine what action to take
		if (tokens[0].compareToIgnoreCase("insert") == 0) {
			Cmd_Insert(tokens);
		} else if (tokens[0].compareToIgnoreCase("remove") == 0) {
			Cmd_Remove(tokens);
		} else if (tokens[0].compareToIgnoreCase("find") == 0) {
			Cmd_Find(tokens);
		} else if (tokens[0].compareToIgnoreCase("search") == 0) {
			Cmd_Search(tokens);
		} else if (tokens[0].compareToIgnoreCase("debug") == 0) {
			Cmd_Debug();
		} else if (tokens[0].compareToIgnoreCase("makenull") == 0) {
			Cmd_Makenull();
		} else {
			System.out.printf(">> \"%s\" command is not supported\n", tokens[0]);
		}
		System.out.printf("\n");
	}
	
	// Parse insert command
	private void Cmd_Insert(String[] tokens) {
		// Check to make sure there is a correct number of arguments for the command
		if (tokens.length != 4) {
			System.out.printf(">> Arguments must be in format \"insert x y name\"\n");
		} else {
			try {
				// Parse and check each argument token
				int x = Integer.parseInt(tokens[1]);
				int y = Integer.parseInt(tokens[2]);
				String name = tokens[3];
				
				if (x < 0 || x > 16383 || y < 0 || y > 16383) {
					System.out.printf(">> Insert failed: Coordinate values out of range\n");
				} else {
					// If all arguments are ok, create a new record with the specified values
					CityRecord newRecord = new CityRecord(name, x, y);
					// Check if record with same coordinates already exists
					LinkedList<Record> list = new LinkedList<Record>();
					CTree.search(x,  y, 0, list);
					if (list.size() != 0) {
						System.out.printf(">> Insert failed: City at (%d,%d) already exists\n", x, y);
					} else {
						// Insert record into both trees
						System.out.printf(">> Inserting city %s (%d,%d)\n", name, x, y);
						GISTree.insert(name, newRecord);
						CTree.insert(newRecord);
					}
				}
			} catch (NumberFormatException e) {
				System.out.printf(">> Insert failed: Unable to parse given coordinates\n");
			}
		}
	}
	
	// Parse remove command
	private void Cmd_Remove(String[] tokens) {
		// Check to make sure there is a correct number of arguments for the command
		if (tokens.length < 2 || tokens.length > 3) {
			System.out.printf(">> Arguments must be in format \"remove x y\" or \" remove name\"\n");
		} else {
			// A single argument means search by name
			if (tokens.length == 2) {
				String name = tokens[1];
				// First check if city exists
				CityRecord record = GISTree.find(name);
				if (record != null) {
					// Remove city if found
					System.out.printf(">> Removing city %s (%d,%d)\n", 
							record.getName(), record.getX(), record.getY());
					GISTree.remove(name, record.getX(), record.getY());
					CTree.remove(record.getX(), record.getY());
				} else {
					System.out.printf(">> Remove failed: City %s not found\n", name);
				}
			// Otherwise search by coordinates
			} else {
				try {
					// Parse and check each argument token
					int x = Integer.parseInt(tokens[1]);
					int y = Integer.parseInt(tokens[2]);
					if (x < 0 || x > 16383 || y < 0 || y > 16383) {
						System.out.printf(">> Remove failed: Coordinate values out of range\n");
					} else {
						// Check if city with coordinates exists
						LinkedList<Record> list = new LinkedList<Record>();
						CTree.search(x,  y, 0, list);
						if (list.size() == 0) {
							System.out.printf(">> Remove failed: City with coordinates (%d,%d) not found\n", x, y);
						} else {
							// Remove using coordinates
							System.out.printf(">> Removing city %s (%d,%d)\n",
									((CityRecord)list.get(0)).getName(), list.get(0).getX(), list.get(0).getY());
							CTree.remove(x, y);
							GISTree.remove(((CityRecord)list.get(0)).getName(), x, y);
						}
					}
				} catch (NumberFormatException e) {
					System.out.printf(">> Remove failed: Unable to parse given coordinates\n");
				}
			}
		}
	}
	
	// Parse find command
	private void Cmd_Find(String[] tokens) {
		// Check to make sure there is a correct number of arguments for the command
		if (tokens.length != 2) {
			System.out.printf(">> Arguments must be in format \"find name\"\n");
		} else {
			String name = tokens[1];
			// Get all records with the matching name
			ArrayList<CityRecord> records = GISTree.findAll(name);
			if (records.size() != 0) {
				// For each city found, print the city details
				for (CityRecord record : records) {
					System.out.printf(">> City found: %s (%d,%d)\n", 
							record.getName(), record.getX(), record.getY());
				}
			} else {
				System.out.printf(">> Find failed: City %s not found\n", name);
			}
		}
	}
	
	// Parse search command
	private void Cmd_Search(String[] tokens) {
		// Check to make sure there is a correct number of arguments for the command
		if (tokens.length != 4) {
			System.out.printf(">> Arguments must be in format \"search x y radius\"\n");
		} else {
			try {
				// Parse and check each argument token
				int x = Integer.parseInt(tokens[1]);
				int y = Integer.parseInt(tokens[2]);
				int r = Integer.parseInt(tokens[3]);
				
				if (r < 0 || r > 16383) {
					System.out.printf(">> Search failed: Radius value out of range\n");
				} else if (x < -16383 || x > 16383 || y < -16383 || y > 16383) {
					System.out.printf(">> Search failed: Coordinate values out of range\n");
				} else {
					// Get a list of all cities found within specified radius of point
					LinkedList<Record> results = new LinkedList<Record>();
					int nodesLookedAt = CTree.search(x, y, r, results);
					System.out.printf(">> %d nodes visited:\n", nodesLookedAt);
					if (results.size() == 0) {
						System.out.printf(">> No records found with specified coordinates\n");
					} else {
						// Print the found cities
						for (Record record : results) {
							System.out.printf(">> %s (%d,%d)\n", ((CityRecord)record).getName(), record.getX(), record.getY());
						}
					}
				}
			} catch (NumberFormatException e) {
				System.out.printf(">> Search failed: Unable to parse given coordinates\n");
			}
		}
	}
	
	// Parse debug command
	private void Cmd_Debug() {
		System.out.printf(">> ");
		CTree.debug(CTree.getRoot());
		System.out.printf("\n");
	}
	
	// Parse makenull command
	private void Cmd_Makenull() {
		GISTree.clear();
		CTree.clear();
		System.out.printf(">> Makenull successful\n");
	}
}
